<?php
/**
 * Review Submission API with Spam Filtering
 * Handles review submissions and saves them to testimonials.json
 */

header('Content-Type: application/json');
require_once __DIR__ . '/../config.php';

// Only accept POST requests
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'message' => 'Method not allowed']);
    exit;
}

// Spam filtering function
function containsSpamKeywords($text) {
    $spamKeywords = [
        // Common spam patterns
        'buy now', 'click here', 'limited time', 'act now', 'urgent',
        'guaranteed', 'risk free', 'no credit check', 'work from home',
        'make money', 'get rich', 'free money', 'cash bonus',
        // Link spam
        'http://', 'https://', 'www.', '.com', '.net', '.org',
        // Suspicious patterns
        'viagra', 'cialis', 'casino', 'poker', 'lottery', 'winner',
        'congratulations', 'you won', 'claim now', 'prize',
        // Email spam
        '@gmail', '@yahoo', '@hotmail', '@outlook',
        // Phone spam
        'call now', 'text now', 'sms', 'whatsapp',
        // Suspicious business terms
        'seo services', 'web design cheap', 'increase traffic',
        'backlinks', 'page rank', 'google ranking'
    ];
    
    $textLower = strtolower($text);
    foreach ($spamKeywords as $keyword) {
        if (strpos($textLower, $keyword) !== false) {
            return true;
        }
    }
    return false;
}

// Rate limiting - check if too many submissions from same IP
function checkRateLimit($ip) {
    $rateLimitFile = __DIR__ . '/../data/review_rate_limit.json';
    $currentTime = time();
    $maxSubmissions = 3; // Max 3 reviews per hour per IP
    $timeWindow = 3600; // 1 hour
    
    if (file_exists($rateLimitFile)) {
        $data = json_decode(file_get_contents($rateLimitFile), true);
    } else {
        $data = [];
    }
    
    // Clean old entries
    $data = array_filter($data, function($entry) use ($currentTime, $timeWindow) {
        return ($currentTime - $entry['time']) < $timeWindow;
    });
    
    // Count submissions from this IP
    $ipSubmissions = array_filter($data, function($entry) use ($ip) {
        return $entry['ip'] === $ip;
    });
    
    if (count($ipSubmissions) >= $maxSubmissions) {
        return false; // Rate limit exceeded
    }
    
    // Add this submission
    $data[] = [
        'ip' => $ip,
        'time' => $currentTime
    ];
    
    file_put_contents($rateLimitFile, json_encode($data, JSON_PRETTY_PRINT));
    return true;
}

// Get and validate input
$input = json_decode(file_get_contents('php://input'), true);

// Honeypot field (should be empty)
if (!empty($input['website']) || !empty($input['url'])) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'Spam detected']);
    exit;
}

// Validate required fields
$required = ['name', 'rating', 'text', 'vehicle'];
foreach ($required as $field) {
    if (empty($input[$field])) {
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => "Field '{$field}' is required"]);
        exit;
    }
}

// Extract and sanitize data
$name = trim($input['name']);
$rating = (int)$input['rating'];
$text = trim($input['text']);
$vehicle = trim($input['vehicle'] ?? '');
$service = trim($input['service'] ?? 'General Service');

// Validate rating
if ($rating < 1 || $rating > 5) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'Rating must be between 1 and 5']);
    exit;
}

// Validate text length
if (strlen($text) < 10) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'Review text must be at least 10 characters']);
    exit;
}

if (strlen($text) > 1000) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'Review text must be less than 1000 characters']);
    exit;
}

// Spam filtering
if (containsSpamKeywords($text) || containsSpamKeywords($name) || containsSpamKeywords($vehicle)) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'Your review contains inappropriate content']);
    exit;
}

// Rate limiting
$ip = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
if (!checkRateLimit($ip)) {
    http_response_code(429);
    echo json_encode(['success' => false, 'message' => 'Too many submissions. Please try again later.']);
    exit;
}

// Generate name initials
$nameParts = explode(' ', $name);
$initials = '';
foreach ($nameParts as $part) {
    if (!empty($part)) {
        $initials .= strtoupper(substr($part, 0, 1)) . '. ';
    }
}
$initials = trim($initials);

// Load existing testimonials
$testimonialsFile = DATA_PATH . 'testimonials.json';
if (!file_exists($testimonialsFile)) {
    http_response_code(500);
    echo json_encode(['success' => false, 'message' => 'Reviews file not found']);
    exit;
}

$testimonialsData = json_decode(file_get_contents($testimonialsFile), true);
if (!isset($testimonialsData['testimonials'])) {
    $testimonialsData['testimonials'] = [];
}

// Get next ID
$maxId = 0;
foreach ($testimonialsData['testimonials'] as $review) {
    if (isset($review['id']) && $review['id'] > $maxId) {
        $maxId = $review['id'];
    }
}
$newId = $maxId + 1;

// Create new review
$newReview = [
    'id' => $newId,
    'name_initials' => $initials,
    'vehicle' => $vehicle,
    'rating' => $rating,
    'text' => $text,
    'date' => date('Y-m-d'),
    'service' => $service,
    'status' => 'pending' // For moderation if needed
];

// Add to testimonials
$testimonialsData['testimonials'][] = $newReview;

// Save to file
if (file_put_contents($testimonialsFile, json_encode($testimonialsData, JSON_PRETTY_PRINT))) {
    // Log successful submission
    error_log("New review submitted: ID {$newId} from IP {$ip}");
    
    echo json_encode([
        'success' => true,
        'message' => 'Thank you for your review! It will be displayed after moderation.',
        'review' => $newReview
    ]);
} else {
    http_response_code(500);
    echo json_encode(['success' => false, 'message' => 'Failed to save review']);
}
?>
