<?php
/**
 * Configuration file for Coventry Mechanic website
 */

// Error logging configuration
error_reporting(E_ALL);
ini_set('display_errors', 0); // Don't display errors to users
ini_set('log_errors', 1); // Enable error logging
ini_set('error_log', __DIR__ . '/errors.log'); // Set error log file

// Site configuration
// Auto-detect site URL from current request (works on any domain)
function get_site_url() {
    try {
        $protocol = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off' || (isset($_SERVER['SERVER_PORT']) && $_SERVER['SERVER_PORT'] == 443)) ? "https://" : "http://";
        $host = $_SERVER['HTTP_HOST'] ?? 'localhost';
        $script_name = isset($_SERVER['SCRIPT_NAME']) ? dirname($_SERVER['SCRIPT_NAME']) : '';
        $base_path = ($script_name === '/' || $script_name === '\\' || $script_name === '.') ? '' : rtrim($script_name, '/');
        return $protocol . $host . $base_path;
    } catch (Exception $e) {
        error_log("Error in get_site_url(): " . $e->getMessage());
        return 'http://localhost';
    }
}

// Set SITE_URL (can be overridden in environment or config)
if (!defined('SITE_URL')) {
    define('SITE_URL', get_site_url());
}

// Site name (can be overridden)
if (!defined('SITE_NAME')) {
    define('SITE_NAME', 'Coventry Mechanic');
}

define('BASE_PATH', __DIR__);

// Data paths
define('DATA_PATH', BASE_PATH . '/data/');

// Load JSON data (with path traversal protection)
function load_json($filename) {
    // Prevent path traversal attacks
    $filename = basename($filename);
    if (!preg_match('/^[a-z0-9_-]+\.json$/', $filename)) {
        return null;
    }
    
    $file = DATA_PATH . $filename;
    // Additional check to ensure file is within data directory
    $real_file = realpath($file);
    $real_data_path = realpath(DATA_PATH);
    
    if ($real_file === false || $real_data_path === false) {
        return null;
    }
    
    if (strpos($real_file, $real_data_path) !== 0) {
        return null; // Path traversal attempt
    }
    
    if (file_exists($file)) {
        $content = file_get_contents($file);
        $data = json_decode($content, true);
        // Check for JSON decode errors
        if (json_last_error() !== JSON_ERROR_NONE) {
            return null;
        }
        return $data;
    }
    return null;
}

// Get current page slug (works with clean URLs and router)
function get_current_slug() {
    // If current_slug is set by router, use it
    if (isset($GLOBALS['current_slug'])) {
        return $GLOBALS['current_slug'];
    }
    
    $uri = $_SERVER['REQUEST_URI'] ?? '/';
    $uri = parse_url($uri, PHP_URL_PATH);
    
    // Remove query string if present
    $uri = strtok($uri, '?');
    
    // Remove base path if in subdirectory
    $script_name = dirname($_SERVER['SCRIPT_NAME'] ?? '');
    if ($script_name !== '/' && $script_name !== '\\' && $script_name !== '.') {
        $uri = str_replace($script_name, '', $uri);
    }
    
    $uri = trim($uri, '/');
    
    // Handle special cases
    if (empty($uri) || $uri === 'index.php' || $uri === 'router.php') {
        return 'home';
    }
    
    // Remove .php extension if present
    $uri = preg_replace('/\.php$/', '', $uri);
    
    return $uri ?: 'home';
}

// Escape output for HTML
function e($string) {
    if ($string === null) return '';
    return htmlspecialchars((string)$string, ENT_QUOTES, 'UTF-8');
}

// Validate and sanitize slug (alphanumeric and hyphens only)
function validate_slug($slug) {
    if (empty($slug)) return false;
    // Only allow lowercase letters, numbers, and hyphens
    return preg_match('/^[a-z0-9-]+$/', $slug) === 1;
}

// Generate CSRF token
function generate_csrf_token() {
    if (session_status() === PHP_SESSION_NONE) {
        session_start();
    }
    if (!isset($_SESSION['csrf_token'])) {
        $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
    }
    return $_SESSION['csrf_token'];
}

// Verify CSRF token
function verify_csrf_token($token) {
    if (session_status() === PHP_SESSION_NONE) {
        session_start();
    }
    return isset($_SESSION['csrf_token']) && hash_equals($_SESSION['csrf_token'], $token);
}

// Rate limiting helper (simple file-based)
function check_rate_limit($identifier, $max_requests = 10, $time_window = 3600) {
    $rate_limit_file = sys_get_temp_dir() . '/rate_limit_' . md5($identifier) . '.txt';
    $current_time = time();
    
    if (file_exists($rate_limit_file)) {
        $data = json_decode(file_get_contents($rate_limit_file), true);
        if ($data && isset($data['time']) && ($current_time - $data['time']) < $time_window) {
            if ($data['count'] >= $max_requests) {
                return false; // Rate limit exceeded
            }
            $data['count']++;
        } else {
            $data = ['time' => $current_time, 'count' => 1];
        }
    } else {
        $data = ['time' => $current_time, 'count' => 1];
    }
    
    file_put_contents($rate_limit_file, json_encode($data));
    return true;
}

// Sanitize input string
function sanitize_input($input, $max_length = 1000) {
    if (!is_string($input)) return '';
    $input = trim($input);
    $input = substr($input, 0, $max_length);
    return strip_tags($input);
}

// Validate email
function validate_email($email) {
    return filter_var($email, FILTER_VALIDATE_EMAIL) !== false;
}

// Validate phone (basic UK format)
function validate_phone($phone) {
    // Remove spaces and common characters
    $phone = preg_replace('/[\s\-\(\)]/', '', $phone);
    // Check if it's a valid UK phone number format
    return preg_match('/^(?:\+44|0)[1-9]\d{8,9}$/', $phone) === 1;
}

// Ping search engines about sitemap updates
function ping_search_engines($sitemap_url) {
    // Only ping in production (not localhost)
    if (strpos(SITE_URL, 'localhost') !== false || 
        strpos(SITE_URL, '127.0.0.1') !== false ||
        strpos(SITE_URL, '192.168.') !== false) {
        return; // Skip pinging on local development
    }
    
    // Rate limiting: Only ping once per day
    $ping_log_file = sys_get_temp_dir() . '/sitemap_ping_' . md5(SITE_URL) . '.txt';
    $last_ping = 0;
    
    if (file_exists($ping_log_file)) {
        $last_ping = (int)file_get_contents($ping_log_file);
    }
    
    // Only ping if last ping was more than 7 days ago
    if (time() - $last_ping < 604800) { // 7 days = 604800 seconds
        return; // Already pinged within last 7 days
    }
    
    // Note: Google and Bing deprecated their ping endpoints (2023-2024)
    // Modern approach: Use IndexNow protocol for Bing/Yandex
    // For Google: Submit sitemap via Google Search Console manually
    
    // IndexNow Protocol (supported by Bing, Yandex, Seznam.cz, Naver, etc.)
    $indexnow_key = substr(md5(SITE_URL), 0, 32);
    $indexnow_endpoints = [
        'https://api.indexnow.org/indexnow',  // Bing IndexNow
        'https://yandex.com/indexnow',        // Yandex IndexNow
    ];
    
    $indexnow_data = [
        'host' => parse_url(SITE_URL, PHP_URL_HOST),
        'key' => $indexnow_key,
        'urlList' => [$sitemap_url]
    ];
    
    // Ping IndexNow endpoints asynchronously (non-blocking)
    foreach ($indexnow_endpoints as $endpoint) {
        if (function_exists('curl_init')) {
            $ch = curl_init($endpoint);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($indexnow_data));
            curl_setopt($ch, CURLOPT_HTTPHEADER, [
                'Content-Type: application/json',
                'User-Agent: IndexNow/1.0'
            ]);
            curl_setopt($ch, CURLOPT_TIMEOUT, 5);
            curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 5);
            curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
            curl_exec($ch);
            $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            curl_close($ch);
            
            // Log successful pings
            if ($http_code == 200 || $http_code == 202) {
                $host = parse_url($endpoint, PHP_URL_HOST);
                error_log("Sitemap notified via IndexNow to {$host}: Success (HTTP {$http_code})");
            }
        }
    }
    
    // Note: For Google, submit sitemap via Google Search Console
    // The sitemap URL is: {$sitemap_url}
    
    // Update last ping time
    file_put_contents($ping_log_file, (string)time());
}

// Format currency
function format_currency($amount) {
    return '£' . number_format($amount, 2);
}

// Get service by slug (with validation)
function get_service_by_slug($slug) {
    // Validate slug format first
    if (!validate_slug($slug)) {
        return null;
    }
    
    $services = load_json('services.json');
    if ($services && isset($services['services'])) {
        foreach ($services['services'] as $service) {
            if (isset($service['slug']) && $service['slug'] === $slug) {
                return $service;
            }
        }
    }
    return null;
}

// Generate canonical URL
function get_canonical_url($path = '') {
    $url = SITE_URL;
    if ($path) {
        $url .= '/' . ltrim($path, '/');
    }
    return rtrim($url, '/');
}

// Generate internal URL (relative or absolute based on context)
function url($path = '') {
    // Use relative URLs for internal links (better for SEO and works on any domain)
    if (empty($path) || $path === '/') {
        return '/';
    }
    return '/' . ltrim($path, '/');
}

// Generate absolute URL when needed (for external sharing, emails, etc.)
function absolute_url($path = '') {
    return get_canonical_url($path);
}

// Load business data (with error handling)
$business = load_json('business.json') ?: [];
$services_data = load_json('services.json') ?: ['services' => []];
$testimonials_data = load_json('testimonials.json') ?: ['testimonials' => []];
$areas_data = load_json('areas.json') ?: [];
$faqs_data = load_json('faqs.json') ?: ['faqs' => []];
$pricing_data = load_json('pricing.json') ?: [];
$pages_data = load_json('pages.json') ?: [];
