<?php
/**
 * Analytics Tracker
 * Tracks website visits, referrals, IPs, user agents, etc.
 */

if (!defined('BASE_PATH')) {
    require_once __DIR__ . '/../config.php';
}

define('ANALYTICS_FILE', DATA_PATH . '/analytics.json');

/**
 * Get client IP address
 * Checks various HTTP headers for real IP (handles proxies/load balancers)
 * @return string Client IP address or 'unknown' if not found
 */
function get_client_ip() {
    $ipKeys = ['HTTP_CLIENT_IP', 'HTTP_X_FORWARDED_FOR', 'HTTP_X_FORWARDED', 'HTTP_X_CLUSTER_CLIENT_IP', 'HTTP_FORWARDED_FOR', 'HTTP_FORWARDED', 'REMOTE_ADDR'];
    
    foreach ($ipKeys as $key) {
        if (array_key_exists($key, $_SERVER) === true) {
            foreach (explode(',', $_SERVER[$key]) as $ip) {
                $ip = trim($ip);
                if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE) !== false) {
                    return $ip;
                }
            }
        }
    }
    
    return $_SERVER['REMOTE_ADDR'] ?? 'unknown';
}

/**
 * Parse user agent string to extract browser and OS information
 * @param string $userAgent User agent string
 * @return array Array with 'browser' and 'os' keys
 */
function parse_user_agent($userAgent) {
    $browser = 'Unknown';
    $os = 'Unknown';
    
    // Browser detection
    if (preg_match('/MSIE|Trident/i', $userAgent)) {
        $browser = 'Internet Explorer';
    } elseif (preg_match('/Edg/i', $userAgent)) {
        $browser = 'Edge';
    } elseif (preg_match('/Chrome/i', $userAgent) && !preg_match('/Edg/i', $userAgent)) {
        $browser = 'Chrome';
    } elseif (preg_match('/Safari/i', $userAgent) && !preg_match('/Chrome/i', $userAgent)) {
        $browser = 'Safari';
    } elseif (preg_match('/Firefox/i', $userAgent)) {
        $browser = 'Firefox';
    } elseif (preg_match('/Opera|OPR/i', $userAgent)) {
        $browser = 'Opera';
    }
    
    // OS detection
    if (preg_match('/Windows NT 10.0/i', $userAgent)) {
        $os = 'Windows 10';
    } elseif (preg_match('/Windows NT 11.0/i', $userAgent)) {
        $os = 'Windows 11';
    } elseif (preg_match('/Windows NT 6.3/i', $userAgent)) {
        $os = 'Windows 8.1';
    } elseif (preg_match('/Windows NT 6.2/i', $userAgent)) {
        $os = 'Windows 8';
    } elseif (preg_match('/Windows NT 6.1/i', $userAgent)) {
        $os = 'Windows 7';
    } elseif (preg_match('/Windows/i', $userAgent)) {
        $os = 'Windows';
    } elseif (preg_match('/Mac OS X/i', $userAgent)) {
        $os = 'macOS';
    } elseif (preg_match('/Linux/i', $userAgent)) {
        $os = 'Linux';
    } elseif (preg_match('/Android/i', $userAgent)) {
        $os = 'Android';
    } elseif (preg_match('/iPhone|iPad|iPod/i', $userAgent)) {
        $os = 'iOS';
    }
    
    return ['browser' => $browser, 'os' => $os];
}

/**
 * Parse referral URL to extract source and search query
 * Detects search engines and extracts search queries
 * @param string $referer Referral URL
 * @return array Array with 'source', 'domain', and optionally 'query' keys
 */
function parse_referral($referer) {
    if (empty($referer)) {
        return ['source' => 'Direct', 'domain' => 'Direct'];
    }
    
    $parsed = parse_url($referer);
    $domain = $parsed['host'] ?? 'Unknown';
    
    // Check if it's a search engine
    $searchEngines = [
        'google' => ['name' => 'Google', 'query_params' => ['q']],
        'bing' => ['name' => 'Bing', 'query_params' => ['q']],
        'yahoo' => ['name' => 'Yahoo', 'query_params' => ['p', 'q']],
        'duckduckgo' => ['name' => 'DuckDuckGo', 'query_params' => ['q']],
        'yandex' => ['name' => 'Yandex', 'query_params' => ['text']],
        'baidu' => ['name' => 'Baidu', 'query_params' => ['wd', 'word']],
        'ask' => ['name' => 'Ask', 'query_params' => ['q']],
        'aol' => ['name' => 'AOL', 'query_params' => ['q', 'query']],
        'ecosia' => ['name' => 'Ecosia', 'query_params' => ['q']],
        'startpage' => ['name' => 'Startpage', 'query_params' => ['query', 'q']],
        'qwant' => ['name' => 'Qwant', 'query_params' => ['q']],
        'brave' => ['name' => 'Brave', 'query_params' => ['q']],
        'swisscows' => ['name' => 'Swisscows', 'query_params' => ['query']]
    ];
    
    $domainLower = strtolower($domain);
    foreach ($searchEngines as $key => $engine) {
        if (strpos($domainLower, $key) !== false) {
            // Extract search query if available
            $query = $parsed['query'] ?? '';
            parse_str($query, $params);
            
            // Try each query parameter for this search engine
            $searchQuery = '';
            foreach ($engine['query_params'] as $param) {
                if (!empty($params[$param])) {
                    $searchQuery = urldecode($params[$param]);
                    break;
                }
            }
            
            return [
                'source' => $engine['name'],
                'domain' => $domain,
                'search_query' => $searchQuery,
                'is_search' => true,
                'search_engine' => $engine['name']
            ];
        }
    }
    
    return [
        'source' => $domain,
        'domain' => $domain,
        'is_search' => false
    ];
}

/**
 * Check if request is from a bot
 * @return bool True if bot, false if real user
 */
function is_bot_request() {
    $userAgent = $_SERVER['HTTP_USER_AGENT'] ?? '';
    if (empty($userAgent)) {
        return true;
    }
    
    $botPatterns = [
        'bot', 'crawl', 'slurp', 'spider', 'facebookexternalhit', 'twitterbot', 
        'linkedinbot', 'whatsapp', 'telegram', 'curl', 'wget', 'python-requests', 
        'scrapy', 'headless', 'phantomjs', 'googlebot', 'bingbot', 'yandexbot'
    ];
    
    $userAgentLower = strtolower($userAgent);
    foreach ($botPatterns as $pattern) {
        if (strpos($userAgentLower, $pattern) !== false) {
            return true;
        }
    }
    
    return false;
}

/**
 * Track a website visit
 * Records visit data including IP, user agent, referrer, and page
 * Excludes admin routes and bot requests
 * @param string $page Page path (default: '/')
 * @param array $additionalData Additional data to include in visit record
 * @return bool True if visit was tracked, false if skipped
 */
function track_visit($page = '/', $additionalData = []) {
    // Skip tracking for admin routes and bot requests
    $requestUri = $_SERVER['REQUEST_URI'] ?? '';
    
    // Exclude admin routes (gsadm)
    if (strpos($requestUri, '/gsadm') === 0 || strpos($page, '/gsadm') === 0) {
        return false;
    }
    
    // Exclude API routes
    if (strpos($requestUri, '/api/') === 0 || strpos($page, '/api/') === 0) {
        return false;
    }
    
    // Exclude bot requests
    if (is_bot_request()) {
        return false;
    }
    
    $ip = get_client_ip();
    $userAgent = $_SERVER['HTTP_USER_AGENT'] ?? 'Unknown';
    $referer = $_SERVER['HTTP_REFERER'] ?? '';
    $timestamp = time();
    $date = date('Y-m-d H:i:s');
    
    $uaInfo = parse_user_agent($userAgent);
    $referralInfo = parse_referral($referer);
    
    $visit = [
        'id' => uniqid('visit_', true),
        'timestamp' => $timestamp,
        'date' => $date,
        'page' => $page,
        'ip' => $ip,
        'user_agent' => $userAgent,
        'browser' => $uaInfo['browser'],
        'os' => $uaInfo['os'],
        'referer' => $referer,
        'referral_source' => $referralInfo['source'],
        'referral_domain' => $referralInfo['domain'],
        'is_search' => $referralInfo['is_search'] ?? false,
        'search_query' => $referralInfo['search_query'] ?? '',
        'search_engine' => $referralInfo['search_engine'] ?? '',
        'additional' => $additionalData
    ];
    
    // Load existing analytics
    $analytics = load_analytics();
    
    // Add visit
    if (!isset($analytics['visits'])) {
        $analytics['visits'] = [];
    }
    
    $analytics['visits'][] = $visit;
    
    // Keep only last 10000 visits to prevent file from growing too large
    if (count($analytics['visits']) > 10000) {
        $analytics['visits'] = array_slice($analytics['visits'], -10000);
    }
    
    // Update statistics
    update_analytics_stats($analytics);
    
    // Save
    return save_analytics($analytics);
}

/**
 * Load analytics data from JSON file
 * Returns default structure if file doesn't exist
 * @return array Analytics data with 'visits' and 'stats' keys
 */
function load_analytics() {
    if (!file_exists(ANALYTICS_FILE)) {
        return [
            'visits' => [],
            'events' => [],
            'stats' => [
                'total_visits' => 0,
                'unique_ips' => [],
                'browsers' => [],
                'referrals' => [],
                'pages' => [],
                'search_queries' => [],
                'os' => []
            ]
        ];
    }
    
    $json = file_get_contents(ANALYTICS_FILE);
    $data = json_decode($json, true) ?: [];
    
    // Ensure events array exists (for backward compatibility)
    if (!isset($data['events'])) {
        $data['events'] = [];
    }
    
    return $data;
}

/**
 * Save analytics data to JSON file
 * @param array $analytics Analytics data array
 * @return int|false Number of bytes written or false on failure
 */
function save_analytics($analytics) {
    return file_put_contents(ANALYTICS_FILE, json_encode($analytics, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES));
}

/**
 * Update analytics statistics from visits data
 * Calculates total visits, unique IPs, browsers, referrals, pages, search queries, and OS
 * @param array &$analytics Analytics data array (passed by reference)
 * @return void
 */
function update_analytics_stats(&$analytics) {
    $visits = $analytics['visits'] ?? [];
    $events = $analytics['events'] ?? [];
    
    $stats = [
        'total_visits' => count($visits),
        'unique_ips' => [],
        'browsers' => [],
        'referrals' => [],
        'pages' => [],
        'search_queries' => [],
        'os' => [],
        'total_clicks' => 0,
        'total_navigations' => 0,
        'total_modal_views' => 0
    ];
    
    // Process events
    foreach ($events as $event) {
        $eventType = $event['event_type'] ?? '';
        if ($eventType === 'click') {
            $stats['total_clicks']++;
        } elseif ($eventType === 'navigation') {
            $stats['total_navigations']++;
        } elseif ($eventType === 'modal_view') {
            $stats['total_modal_views']++;
        }
    }
    
    foreach ($visits as $visit) {
        // Unique IPs
        if (!in_array($visit['ip'], $stats['unique_ips'])) {
            $stats['unique_ips'][] = $visit['ip'];
        }
        
        // Browsers
        $browser = $visit['browser'] ?? 'Unknown';
        $stats['browsers'][$browser] = ($stats['browsers'][$browser] ?? 0) + 1;
        
        // OS
        $os = $visit['os'] ?? 'Unknown';
        $stats['os'][$os] = ($stats['os'][$os] ?? 0) + 1;
        
        // Referrals
        $refSource = $visit['referral_source'] ?? 'Direct';
        $stats['referrals'][$refSource] = ($stats['referrals'][$refSource] ?? 0) + 1;
        
        // Pages
        $page = $visit['page'] ?? '/';
        $stats['pages'][$page] = ($stats['pages'][$page] ?? 0) + 1;
        
        // Search engine queries
        if (isset($visit['is_search']) && $visit['is_search']) {
            $query = $visit['search_query'] ?? '';
            $engine = $visit['search_engine'] ?? $visit['referral_source'] ?? 'Unknown';
            if (!empty($query)) {
                $key = $engine . ': ' . $query;
                $stats['search_queries'][$key] = ($stats['search_queries'][$key] ?? 0) + 1;
            }
        }
    }
    
    $analytics['stats'] = $stats;
}

/**
 * Get analytics statistics
 * @param int $days Number of days to analyze (default: 30)
 * @return array Statistics array
 */
function get_analytics_stats($days = 30) {
    $analytics = load_analytics();
    $visits = $analytics['visits'] ?? [];
    $events = $analytics['events'] ?? [];
    
    $cutoffTime = time() - ($days * 24 * 60 * 60);
    $recentVisits = array_filter($visits, function($visit) use ($cutoffTime) {
        return ($visit['timestamp'] ?? 0) >= $cutoffTime;
    });
    
    $recentEvents = array_filter($events, function($event) use ($cutoffTime) {
        return ($event['timestamp'] ?? 0) >= $cutoffTime;
    });
    
    $stats = [
        'total_visits' => count($recentVisits),
        'unique_ips' => count(array_unique(array_column($recentVisits, 'ip'))),
        'browsers' => [],
        'referrals' => [],
        'pages' => [],
        'search_queries' => [],
        'os' => [],
        'daily_visits' => [],
        'total_clicks' => 0,
        'total_navigations' => 0,
        'total_modal_views' => 0,
        'users_with_clicks' => 0,
        'users_with_navigation' => 0
    ];
    
    // Process events
    $uniqueIpsWithClicks = [];
    $uniqueIpsWithNavigation = [];
    foreach ($recentEvents as $event) {
        $eventType = $event['event_type'] ?? '';
        $ip = $event['ip'] ?? '';
        
        if ($eventType === 'click') {
            $stats['total_clicks']++;
            if (!in_array($ip, $uniqueIpsWithClicks)) {
                $uniqueIpsWithClicks[] = $ip;
            }
        } elseif ($eventType === 'navigation') {
            $stats['total_navigations']++;
            if (!in_array($ip, $uniqueIpsWithNavigation)) {
                $uniqueIpsWithNavigation[] = $ip;
            }
        } elseif ($eventType === 'modal_view') {
            $stats['total_modal_views']++;
        }
    }
    
    $stats['users_with_clicks'] = count($uniqueIpsWithClicks);
    $stats['users_with_navigation'] = count($uniqueIpsWithNavigation);
    
    foreach ($recentVisits as $visit) {
        // Browsers
        $browser = $visit['browser'] ?? 'Unknown';
        $stats['browsers'][$browser] = ($stats['browsers'][$browser] ?? 0) + 1;
        
        // OS
        $os = $visit['os'] ?? 'Unknown';
        $stats['os'][$os] = ($stats['os'][$os] ?? 0) + 1;
        
        // Referrals
        $refSource = $visit['referral_source'] ?? 'Direct';
        $stats['referrals'][$refSource] = ($stats['referrals'][$refSource] ?? 0) + 1;
        
        // Pages
        $page = $visit['page'] ?? '/';
        $stats['pages'][$page] = ($stats['pages'][$page] ?? 0) + 1;
        
        // Search engine queries
        if (isset($visit['is_search']) && $visit['is_search']) {
            $query = $visit['search_query'] ?? '';
            $engine = $visit['search_engine'] ?? $visit['referral_source'] ?? 'Unknown';
            if (!empty($query)) {
                $key = $engine . ': ' . $query;
                $stats['search_queries'][$key] = ($stats['search_queries'][$key] ?? 0) + 1;
            }
        }
        
        // Daily visits
        $date = date('Y-m-d', $visit['timestamp'] ?? time());
        $stats['daily_visits'][$date] = ($stats['daily_visits'][$date] ?? 0) + 1;
    }
    
    // Sort arrays
    if (is_array($stats['browsers'])) arsort($stats['browsers']);
    if (is_array($stats['referrals'])) arsort($stats['referrals']);
    if (is_array($stats['pages'])) arsort($stats['pages']);
    if (is_array($stats['search_queries'])) arsort($stats['search_queries']);
    if (is_array($stats['os'])) arsort($stats['os']);
    if (is_array($stats['daily_visits'])) ksort($stats['daily_visits']);
    
    return $stats;
}

/**
 * Delete analytics visit by ID
 * Removes visit from analytics and updates statistics
 * @param string $visitId Visit ID to delete
 * @return int|false Number of bytes written or false on failure
 */
function delete_analytics_visit($visitId) {
    $analytics = load_analytics();
    $visits = $analytics['visits'] ?? [];
    
    $analytics['visits'] = array_filter($visits, function($visit) use ($visitId) {
        return ($visit['id'] ?? '') !== $visitId;
    });
    
    $analytics['visits'] = array_values($analytics['visits']);
    update_analytics_stats($analytics);
    
    return save_analytics($analytics);
}

/**
 * Clear all analytics data
 * Resets visits and statistics to empty state
 * @return int|false Number of bytes written or false on failure
 */
function clear_analytics() {
    $analytics = [
        'visits' => [],
        'stats' => [
            'total_visits' => 0,
            'unique_ips' => [],
            'browsers' => [],
            'referrals' => [],
            'pages' => [],
            'search_queries' => [],
            'os' => []
        ]
    ];
    return save_analytics($analytics);
}
