# URL Routing & SEO-Friendly URLs Guide

This document explains how URL routing works in the template and how to ensure SEO-friendly URLs work on any domain.

## How It Works

### Auto-Domain Detection

The site automatically detects the current domain from the HTTP request, so it works on:
- `localhost` (development)
- `example.com` (production)
- `subdomain.example.com` (subdomains)
- Any custom domain

**Implementation:** `config.php` includes `get_site_url()` function that auto-detects the protocol and domain.

### URL Structure

All URLs are SEO-friendly and work without `.php` extensions:

#### Main Pages
- `/` → `index.php` (Homepage)
- `/services` → `services.php`
- `/pricing` → `pricing.php`
- `/about` → `about.php`
- `/contact` → `contact.php`
- `/reviews` → `reviews.php`
- `/faq` → `faq.php`
- `/sitemap` → `sitemap.php`

#### Service Pages
- `/services/mot-coventry` → `service-detail.php?slug=mot-coventry`
- `/services/car-servicing-coventry` → `service-detail.php?slug=car-servicing-coventry`
- `/services/brakes` → `service-detail.php?slug=brakes`

#### Policy Pages
- `/privacy-policy` → `privacy-policy.php`
- `/cookie-policy` → `cookie-policy.php`
- `/terms` → `terms.php`

#### Special URLs
- `/sitemap.xml` → `sitemap.xml.php` (XML sitemap for search engines)
- `/api/quote.php` → Direct API endpoint (no rewriting)

## .htaccess Rules

### Rule Order (Important!)

Rules are processed in order:

1. **API Endpoints** - Allow direct access to API files
2. **Service Detail Pages** - Rewrite `/services/slug` to `service-detail.php?slug=slug`
3. **Sitemap XML** - Rewrite `/sitemap.xml` to `sitemap.xml.php`
4. **Remove .php Extension** - Rewrite `/page` to `page.php`
5. **Remove Trailing Slash** - Redirect `/page/` to `/page`

### Key Rules Explained

```apache
# Service detail pages - SEO friendly URLs
RewriteCond %{REQUEST_FILENAME} !-f
RewriteCond %{REQUEST_FILENAME} !-d
RewriteRule ^services/([a-z0-9-]+)/?$ service-detail.php?slug=$1 [L,QSA]
```

This rule:
- Only applies if the requested file/directory doesn't exist
- Matches `/services/slug` pattern
- Rewrites to `service-detail.php?slug=slug`
- `[L,QSA]` = Last rule, Query String Append

```apache
# Remove .php extension
RewriteCond %{REQUEST_FILENAME} !-d
RewriteCond %{REQUEST_FILENAME}\.php -f
RewriteRule ^([^\.]+)$ $1.php [NC,L]
```

This rule:
- Only applies if it's not a directory
- Checks if corresponding `.php` file exists
- Rewrites `/page` to `page.php` internally

## URL Generation Functions

### `url($path)` - Relative URLs (Recommended)

Use for internal links - works on any domain:

```php
<a href="<?php echo url('services'); ?>">Services</a>
// Outputs: /services

<a href="<?php echo url('services/mot-coventry'); ?>">MOT</a>
// Outputs: /services/mot-coventry
```

### `absolute_url($path)` - Absolute URLs

Use for:
- Canonical URLs
- Open Graph tags
- Schema.org markup
- External sharing

```php
<link rel="canonical" href="<?php echo absolute_url('services/mot-coventry'); ?>">
// Outputs: https://yourdomain.com/services/mot-coventry
```

### `get_canonical_url($path)` - Canonical URLs

Same as `absolute_url()`, used for SEO canonical tags.

## Internal Links

**Best Practice:** Use relative URLs (starting with `/`) for internal links:

```php
<!-- Good - works on any domain -->
<a href="/services">Services</a>
<a href="/contact">Contact</a>
<a href="/services/mot-coventry">MOT</a>

<!-- Also good - using url() function -->
<a href="<?php echo url('services'); ?>">Services</a>
```

**Avoid:** Hardcoded absolute URLs in templates:

```php
<!-- Bad - breaks on different domains -->
<a href="https://coventrymechanic.co.uk/services">Services</a>
```

## Testing Routes

### Manual Testing Checklist

1. **Homepage**
   - `/` → Should show homepage
   - `/index.php` → Should redirect to `/` (or show same content)

2. **Main Pages**
   - `/services` → Should show services page
   - `/services/` → Should redirect to `/services` (no trailing slash)

3. **Service Pages**
   - `/services/mot-coventry` → Should show MOT service page
   - `/services/invalid-slug` → Should show 404 page

4. **API Endpoints**
   - `/api/quote.php` → Should work (POST requests)

5. **Sitemap**
   - `/sitemap.xml` → Should show XML sitemap
   - `/sitemap` → Should show HTML sitemap

### Common Issues & Solutions

#### Issue: 404 on all pages
**Solution:** Check if `mod_rewrite` is enabled:
```apache
# In Apache config or .htaccess
LoadModule rewrite_module modules/mod_rewrite.so
```

#### Issue: URLs show `.php` extension
**Solution:** Check `.htaccess` is being read:
- Ensure `.htaccess` file exists in root
- Check Apache `AllowOverride` is set to `All` or `FileInfo`
- Verify file permissions

#### Issue: Service pages return 404
**Solution:** 
- Check slug format (lowercase, hyphens only)
- Verify service exists in `data/services.json`
- Check `.htaccess` rewrite rules are correct

#### Issue: API endpoint not working
**Solution:**
- Verify `/api/quote.php` file exists
- Check file permissions (should be readable)
- Ensure `.htaccess` allows API access
- Check browser console for CORS errors

## Domain-Specific Configuration

### Override SITE_URL (Optional)

If you need to override auto-detection, add to `config.php` before the auto-detection:

```php
// Override auto-detection (optional)
define('SITE_URL', 'https://yourdomain.com');
```

### Subdirectory Installation

If installing in a subdirectory (e.g., `/garage/`), update `.htaccess`:

```apache
RewriteBase /garage/
```

And update `get_site_url()` in `config.php` to include subdirectory.

## SEO Considerations

### Canonical URLs

Every page includes a canonical URL to prevent duplicate content:

```html
<link rel="canonical" href="https://yourdomain.com/services/mot-coventry">
```

### Sitemap

The sitemap automatically includes all pages:
- Main pages
- Service pages
- Policy pages

Access at: `https://yourdomain.com/sitemap.xml`

### URL Structure Best Practices

✅ **Good URLs:**
- `/services/mot-coventry`
- `/services/car-servicing`
- `/contact`

❌ **Bad URLs:**
- `/services.php?slug=mot-coventry`
- `/page.php?id=123`
- `/services/mot_coventry` (underscores)

## Development vs Production

### Development (localhost)
- Works on `http://localhost/coventry-mechanic/`
- Works on `http://127.0.0.1/coventry-mechanic/`
- Auto-detects protocol and path

### Production
- Works on any domain
- Auto-detects HTTPS if available
- Works with or without `www` prefix

## Troubleshooting

### Enable Rewrite Logging (Debug)

Add to `.htaccess` for debugging:

```apache
RewriteLog /path/to/rewrite.log
RewriteLogLevel 3
```

### Check Current URL

Add this to any page for debugging:

```php
<?php
echo "Current URL: " . SITE_URL . $_SERVER['REQUEST_URI'];
echo "<br>Canonical: " . get_canonical_url();
?>
```

---

**Last Updated:** April 2024
