# Security Implementation

This document outlines the security measures implemented in the Coventry Mechanic website template.

## Security Features

### 1. XSS (Cross-Site Scripting) Prevention

- **Output Escaping**: All user-generated content is escaped using the `e()` function which uses `htmlspecialchars()` with `ENT_QUOTES` and `UTF-8` encoding
- **JSON Encoding**: All JSON responses use `JSON_HEX_TAG | JSON_HEX_AMP | JSON_HEX_APOS | JSON_HEX_QUOT` flags to prevent XSS in JSON contexts
- **Input Sanitization**: All user inputs are sanitized using `sanitize_input()` function which strips tags and limits length

**Implementation:**
- `config.php`: `e()` function for HTML output escaping
- All template files use `e()` for dynamic content
- JSON API responses properly encoded

### 2. CSRF (Cross-Site Request Forgery) Protection

- **Token Generation**: CSRF tokens generated using `bin2hex(random_bytes(32))` for cryptographically secure randomness
- **Session-Based Storage**: Tokens stored in PHP sessions
- **Token Verification**: All forms verify tokens using `hash_equals()` for timing-safe comparison
- **Token Regeneration**: CSRF token regenerated after successful form submission

**Implementation:**
- `config.php`: `generate_csrf_token()` and `verify_csrf_token()` functions
- `contact.php`: CSRF token included in all forms

### 3. SQL Injection Prevention

- **No Database**: This template uses JSON files instead of a database, eliminating SQL injection risks
- **File-Based Storage**: All data stored in JSON files with path traversal protection

### 4. Path Traversal Prevention

- **Filename Validation**: `load_json()` function validates filenames using `basename()` and regex pattern matching
- **Realpath Checking**: Verifies file paths are within the data directory using `realpath()` comparison
- **Slug Validation**: Service slugs validated using `validate_slug()` function with regex pattern `/^[a-z0-9-]+$/`

**Implementation:**
- `config.php`: `load_json()` includes path traversal checks
- `service-detail.php`: Slug validation before processing

### 5. Input Validation & Whitelisting

- **Whitelist Validation**: API endpoints use strict whitelists for allowed values
- **Type Checking**: All inputs validated against expected types
- **Length Limits**: Input fields have maximum length restrictions
- **Format Validation**: Email addresses validated using `filter_var()`, phone numbers with regex

**Implementation:**
- `api/quote.php`: Whitelist arrays for service, vehicle, engine, urgency
- `contact.php`: Email validation, length checks, format validation

### 6. Rate Limiting

- **File-Based Rate Limiting**: Simple but effective rate limiting using temporary files
- **Per-IP Tracking**: Rate limits tracked by client IP address
- **Configurable Limits**: 
  - Contact form: 5 requests per hour
  - Quote API: 20 requests per hour

**Implementation:**
- `config.php`: `check_rate_limit()` function
- `contact.php`: Rate limiting on form submissions
- `api/quote.php`: Rate limiting on API requests

### 7. Insecure Direct Object References

- **Slug Validation**: All service slugs validated before lookup
- **Existence Checking**: Services verified to exist before display
- **404 Handling**: Invalid slugs return 404 page instead of error

**Implementation:**
- `service-detail.php`: Slug validation and service existence check
- `config.php`: `get_service_by_slug()` validates slug format

### 8. Security Headers

HTTP security headers configured in `.htaccess`:

- **X-Content-Type-Options**: `nosniff` - Prevents MIME type sniffing
- **X-Frame-Options**: `SAMEORIGIN` - Prevents clickjacking
- **X-XSS-Protection**: `1; mode=block` - Enables XSS filter
- **Referrer-Policy**: `strict-origin-when-cross-origin` - Controls referrer information
- **Permissions-Policy**: Restricts browser features
- **Strict-Transport-Security**: Forces HTTPS (when enabled)
- **Content-Security-Policy**: Restricts resource loading

### 9. Honeypot Spam Protection

- **Hidden Field**: Contact form includes hidden "website" field
- **Bot Detection**: Bots filling honeypot field are silently rejected
- **User Experience**: Legitimate users never see or interact with honeypot

**Implementation:**
- `contact.php`: Honeypot field in form, validation in processing

### 10. File Operation Security

- **Path Validation**: All file operations validate paths
- **Directory Traversal Prevention**: Realpath checks ensure files stay within allowed directories
- **Error Handling**: File operations use `@` operator with proper error handling
- **Lock Files**: Rate limit files use file locking where appropriate

### 11. Session Security

- **Session Configuration**: PHP sessions used for CSRF tokens
- **Token Storage**: Tokens stored server-side in sessions
- **Secure Comparison**: Token verification uses `hash_equals()` for timing-safe comparison

### 12. Error Handling

- **No Information Leakage**: Error messages don't reveal system information
- **Generic Messages**: User-facing errors are generic and helpful
- **Logging**: Security events can be logged (contact form submissions)

## Mobile Responsiveness

### Responsive Design

- **Mobile-First**: All pages use Tailwind CSS mobile-first approach
- **Breakpoints**: 
  - `sm:` - 640px and up
  - `md:` - 768px and up
  - `lg:` - 1024px and up
- **Flexible Layouts**: Grid and flexbox layouts adapt to screen size
- **Touch-Friendly**: Buttons and links sized appropriately for touch

### Viewport Configuration

```html
<meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=5.0, user-scalable=yes">
```

- Allows user scaling for accessibility
- Prevents zoom issues on mobile devices
- Format detection for phone numbers

## Security Best Practices

### For Developers

1. **Never Trust User Input**: Always validate and sanitize
2. **Use Prepared Statements**: If adding database later, use PDO prepared statements
3. **Keep Dependencies Updated**: Regularly update PHP and server software
4. **Regular Security Audits**: Review code for security issues
5. **Monitor Logs**: Check contact logs and server logs regularly

### For Deployment

1. **Enable HTTPS**: Uncomment HTTPS redirect in `.htaccess`
2. **Secure File Permissions**: Set appropriate file permissions (644 for files, 755 for directories)
3. **Disable Directory Listing**: Already configured in `.htaccess`
4. **Regular Backups**: Backup JSON data files regularly
5. **Server Hardening**: Follow server security best practices

## Known Limitations

1. **Simple Rate Limiting**: Current rate limiting is file-based and may not scale to high traffic
2. **No Database**: JSON files work well for small sites but may need database for larger scale
3. **Email Sending**: Contact form doesn't send emails - needs implementation
4. **Session Storage**: Sessions use default PHP session storage - consider Redis/Memcached for production

## Reporting Security Issues

If you discover a security vulnerability, please:
1. Do not disclose publicly
2. Contact the development team
3. Provide detailed information about the issue
4. Allow time for fix before disclosure

## Security Checklist

- [x] XSS prevention (output escaping)
- [x] CSRF protection (tokens)
- [x] Input validation (whitelisting)
- [x] Path traversal prevention
- [x] Rate limiting
- [x] Security headers
- [x] Honeypot spam protection
- [x] Mobile responsive design
- [x] Secure file operations
- [x] Error handling without information leakage

---

**Last Updated**: April 2024
