# Troubleshooting Guide

## Internal Server Error (500)

If you're seeing an Internal Server Error, follow these steps:

### 1. Check Error Logs

**XAMPP (Windows):**
- Check: `C:\xampp\apache\logs\error.log`
- Or: `C:\xampp\php\logs\php_error_log`

**Linux:**
- Check: `/var/log/apache2/error.log`
- Or: `/var/log/php/error.log`

### 2. Common Issues & Fixes

#### Issue: Missing JSON Files
**Symptom:** Error about undefined variables like `$business`

**Fix:**
- Ensure all JSON files exist in `data/` directory:
  - `data/business.json`
  - `data/services.json`
  - `data/testimonials.json`
  - `data/faqs.json`
  - `data/pricing.json`
  - `data/areas.json`
  - `data/pages.json`

#### Issue: Invalid JSON Syntax
**Symptom:** JSON decode errors

**Fix:**
- Validate JSON files at: https://jsonlint.com/
- Check for:
  - Trailing commas
  - Missing quotes
  - Unclosed brackets/braces

#### Issue: Router Errors
**Symptom:** Routes not working, 500 errors

**Fix:**
- Check `routes.php` exists
- Verify `router.php` can access `config.php`
- Check file permissions

#### Issue: Missing Functions
**Symptom:** "Call to undefined function" errors

**Fix:**
- Ensure `config.php` is included before using functions
- Check PHP version (requires 7.4+)

### 3. Quick Diagnostic Script

Access `/error-check.php` to check for:
- PHP syntax errors
- Missing functions
- Missing constants
- File existence

### 4. Enable Error Display (Development Only)

**Temporary fix for debugging:**

Edit `router.php` or `config.php`:

```php
ini_set('display_errors', 1);
error_reporting(E_ALL);
```

**⚠️ Remove this in production!**

### 5. Check File Permissions

**Required permissions:**
- PHP files: 644 (readable)
- Directories: 755 (executable)
- `data/` directory: 755 (readable)

### 6. Verify .htaccess

**Check if mod_rewrite is enabled:**

Create `test-rewrite.php`:
```php
<?php
if (function_exists('apache_get_modules')) {
    echo in_array('mod_rewrite', apache_get_modules()) ? 'Enabled' : 'Disabled';
} else {
    echo 'Cannot check (not Apache)';
}
?>
```

### 7. Common Error Messages

#### "Call to undefined function"
- **Cause:** Function not loaded
- **Fix:** Check `config.php` is included

#### "Undefined variable: $business"
- **Cause:** JSON file not loaded
- **Fix:** Check `data/business.json` exists and is valid

#### "Routes configuration file not found"
- **Cause:** `routes.php` missing
- **Fix:** Ensure `routes.php` exists in root

#### "Invalid routes configuration"
- **Cause:** `routes.php` doesn't return array
- **Fix:** Check `routes.php` syntax

### 8. Step-by-Step Debugging

1. **Test config.php:**
   ```php
   <?php require_once 'config.php'; echo 'Config loaded'; ?>
   ```

2. **Test routes.php:**
   ```php
   <?php $routes = require 'routes.php'; var_dump($routes); ?>
   ```

3. **Test router.php:**
   - Access any URL
   - Check error logs
   - Verify route matching

4. **Test JSON loading:**
   ```php
   <?php require_once 'config.php'; var_dump(load_json('business.json')); ?>
   ```

### 9. Production Checklist

Before going live:
- [ ] Remove `error-check.php`
- [ ] Remove `test-routes.php` or secure it
- [ ] Set `display_errors = 0` in PHP
- [ ] Check all JSON files are valid
- [ ] Verify `.htaccess` is working
- [ ] Test all routes
- [ ] Check error logs are being written

### 10. Still Having Issues?

1. **Check PHP version:**
   ```php
   <?php echo phpversion(); ?>
   ```
   Requires PHP 7.4+

2. **Check required extensions:**
   - JSON (usually built-in)
   - cURL (for sitemap pinging)

3. **Check server configuration:**
   - Apache `AllowOverride` set to `All`
   - `mod_rewrite` enabled
   - PHP error logging enabled

---

**Last Updated:** April 2024
