<?php
require_once __DIR__ . '/../config.php';
$current_slug = 'contact';

// Start session for CSRF token
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

$form_submitted = false;
$form_error = '';
$form_success = false;
$csrf_token = generate_csrf_token();

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Rate limiting
    $client_ip = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
    if (!check_rate_limit('contact_' . $client_ip, 5, 3600)) {
        $form_error = 'Too many submissions. Please try again later.';
    } else {
        // CSRF token validation
        $submitted_token = $_POST['csrf_token'] ?? '';
        if (!verify_csrf_token($submitted_token)) {
            $form_error = 'Invalid security token. Please refresh the page and try again.';
        } else {
            // Sanitize and validate inputs
            $name = sanitize_input($_POST['name'] ?? '', 100);
            $email = sanitize_input($_POST['email'] ?? '', 255);
            $phone = sanitize_input($_POST['phone'] ?? '', 20);
            $service = sanitize_input($_POST['service'] ?? '', 100);
            $message = sanitize_input($_POST['message'] ?? '', 2000);
            $honeypot = $_POST['website'] ?? ''; // Honeypot field
            
            // Honeypot check
            if (!empty($honeypot)) {
                // Bot detected, silently fail
                $form_success = true; // Pretend success to bot
            } elseif (empty($name) || empty($email) || empty($message)) {
                $form_error = 'Please fill in all required fields.';
            } elseif (!validate_email($email)) {
                $form_error = 'Please enter a valid email address.';
            } elseif (strlen($name) < 2 || strlen($name) > 100) {
                $form_error = 'Name must be between 2 and 100 characters.';
            } elseif (strlen($message) < 10 || strlen($message) > 2000) {
                $form_error = 'Message must be between 10 and 2000 characters.';
            } else {
                // Validate service slug if provided
                if (!empty($service) && !validate_slug($service)) {
                    $service = ''; // Reset invalid service
                }
                
                // This is a demo - no email is sent
                // Just mark as successful to show pop-up
                $form_success = true;
                $form_submitted = true;
                
                // Regenerate CSRF token after successful submission
                $csrf_token = generate_csrf_token();
            }
        }
    }
}
?>
<?php include __DIR__ . '/../partials/header.php'; ?>
<?php include __DIR__ . '/../partials/breadcrumbs.php'; ?>

<section class="py-16 bg-white">
    <div class="container mx-auto px-4">
        <div class="max-w-6xl mx-auto">
            <h1 class="text-4xl font-bold text-center mb-4">Contact Us</h1>
            <p class="text-center text-gray-600 mb-12">
                Get in touch for a quote, to book an appointment, or with any questions
            </p>
            
            <!-- Demo Notice -->
            <div id="demo-notice-modal" class="fixed inset-0 bg-black/60 backdrop-blur-sm z-50 hidden items-center justify-center p-4" style="display: none;">
                <div class="bg-white rounded-3xl shadow-2xl max-w-md w-full p-8 transform transition-all border border-gray-200">
                    <div class="text-center">
                        <div class="w-16 h-16 bg-gradient-to-r from-indigo-500 to-purple-500 rounded-full flex items-center justify-center mx-auto mb-4">
                            <svg class="w-8 h-8 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M13 16h-1v-4h-1m1-4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                            </svg>
                        </div>
                        <h3 class="text-2xl font-bold text-gray-900 mb-3">This is a Demo</h3>
                        <p class="text-gray-700 mb-6 leading-relaxed">
                            No email has been transmitted. This is a demonstration website and the contact form is for display purposes only.
                        </p>
                        <button onclick="closeDemoNotice()" 
                                class="w-full bg-gradient-to-r from-indigo-600 to-purple-600 text-white px-6 py-3 rounded-xl font-semibold hover:from-indigo-700 hover:to-purple-700 transition-all duration-300 transform hover:scale-105">
                            Understood
                        </button>
                    </div>
                </div>
            </div>
            
            <div class="grid grid-cols-1 lg:grid-cols-2 gap-12">
                <!-- Contact Form -->
                <div>
                    <h2 class="text-2xl font-bold mb-6">Send us a Message</h2>
                    
                    <?php if ($form_error): ?>
                    <div class="bg-red-50 border border-red-200 text-red-800 px-4 py-3 rounded-lg mb-6">
                        <p><?php echo e($form_error); ?></p>
                    </div>
                    <?php endif; ?>
                    
                    <form method="POST" action="/contact" id="contact-form" class="space-y-4">
                        <!-- CSRF Token -->
                        <input type="hidden" name="csrf_token" value="<?php echo e($csrf_token); ?>">
                        
                        <div>
                            <label for="name" class="block text-sm font-medium text-gray-700 mb-2">Name *</label>
                            <input type="text" id="name" name="name" required maxlength="100"
                                   value="<?php echo e($_POST['name'] ?? ''); ?>"
                                   class="w-full border-2 border-gray-200 rounded-xl px-4 py-3 focus:ring-2 focus:ring-purple-500 focus:border-purple-500 transition-all">
                        </div>
                        
                        <div>
                            <label for="email" class="block text-sm font-medium text-gray-700 mb-2">Email *</label>
                            <input type="email" id="email" name="email" required maxlength="255"
                                   value="<?php echo e($_POST['email'] ?? ''); ?>"
                                   class="w-full border-2 border-gray-200 rounded-xl px-4 py-3 focus:ring-2 focus:ring-purple-500 focus:border-purple-500 transition-all">
                        </div>
                        
                        <div>
                            <label for="phone" class="block text-sm font-medium text-gray-700 mb-2">Phone</label>
                            <input type="tel" id="phone" name="phone" maxlength="20"
                                   value="<?php echo e($_POST['phone'] ?? ''); ?>"
                                   class="w-full border-2 border-gray-200 rounded-xl px-4 py-3 focus:ring-2 focus:ring-purple-500 focus:border-purple-500 transition-all">
                        </div>
                        
                        <div>
                            <label for="service" class="block text-sm font-medium text-gray-700 mb-2">Service Interested In</label>
                            <select id="service" name="service"
                                    class="w-full border-2 border-gray-200 rounded-xl px-4 py-3 focus:ring-2 focus:ring-purple-500 focus:border-purple-500 transition-all">
                                <option value="">Select a service...</option>
                                <?php foreach ($services_data['services'] as $service_item): ?>
                                <option value="<?php echo e($service_item['slug']); ?>" <?php echo (isset($_POST['service']) && $_POST['service'] === $service_item['slug']) ? 'selected' : ''; ?>>
                                    <?php echo e($service_item['title']); ?>
                                </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        
                        <div>
                            <label for="message" class="block text-sm font-medium text-gray-700 mb-2">Message *</label>
                            <textarea id="message" name="message" rows="5" required maxlength="2000"
                                      class="w-full border-2 border-gray-200 rounded-xl px-4 py-3 focus:ring-2 focus:ring-purple-500 focus:border-purple-500 transition-all resize-none"><?php echo e($_POST['message'] ?? ''); ?></textarea>
                        </div>
                        
                        <!-- Honeypot field -->
                        <div style="position: absolute; left: -9999px;" aria-hidden="true">
                            <label for="website">Website</label>
                            <input type="text" id="website" name="website" tabindex="-1" autocomplete="off">
                        </div>
                        
                        <button type="submit" class="w-full bg-gradient-to-r from-indigo-600 via-purple-600 to-pink-600 text-white px-6 py-4 rounded-xl font-bold hover:from-indigo-700 hover:via-purple-700 hover:to-pink-700 shadow-lg hover:shadow-xl transition-all duration-300 transform hover:scale-105">
                            Send Message
                        </button>
                    </form>
                    
                    <script>
                    // Show demo notice on form submission
                    document.getElementById('contact-form')?.addEventListener('submit', function(e) {
                        e.preventDefault();
                        
                        // Validate form
                        const name = document.getElementById('name').value.trim();
                        const email = document.getElementById('email').value.trim();
                        const message = document.getElementById('message').value.trim();
                        
                        if (!name || !email || !message) {
                            alert('Please fill in all required fields.');
                            return;
                        }
                        
                        // Validate email format
                        const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
                        if (!emailRegex.test(email)) {
                            alert('Please enter a valid email address.');
                            return;
                        }
                        
                        // Show demo notice
                        showDemoNotice();
                        // Reset form
                        this.reset();
                    });
                    
                    function showDemoNotice() {
                        const modal = document.getElementById('demo-notice-modal');
                        if (modal) {
                            modal.classList.remove('hidden');
                            modal.style.display = 'flex';
                            document.body.style.overflow = 'hidden';
                        }
                    }
                    
                    function closeDemoNotice() {
                        const modal = document.getElementById('demo-notice-modal');
                        if (modal) {
                            modal.classList.add('hidden');
                            setTimeout(() => {
                                modal.style.display = 'none';
                                document.body.style.overflow = '';
                            }, 300);
                        }
                    }
                    
                    // Close modal when clicking outside
                    document.getElementById('demo-notice-modal')?.addEventListener('click', function(e) {
                        if (e.target === this) {
                            closeDemoNotice();
                        }
                    });
                    
                    // Close modal with Escape key
                    document.addEventListener('keydown', function(e) {
                        if (e.key === 'Escape') {
                            const modal = document.getElementById('demo-notice-modal');
                            if (modal && modal.style.display !== 'none') {
                                closeDemoNotice();
                            }
                        }
                    });
                    </script>
                </div>
                
                <!-- Contact Information -->
                <div>
                    <h2 class="text-2xl font-bold mb-6">Get in Touch</h2>
                    
                    <div class="space-y-6 mb-8">
                        <div>
                            <h3 class="font-semibold text-lg mb-2">Phone</h3>
                            <a href="tel:<?php echo e($business['contact']['phone_formatted']); ?>" 
                               class="text-blue-600 hover:text-blue-700 text-lg">
                                <?php echo e($business['contact']['phone']); ?>
                            </a>
                        </div>
                        
                        <div>
                            <h3 class="font-semibold text-lg mb-2">Contact Us</h3>
                            <p class="text-gray-700">
                                Please use the contact form to get in touch with us. We'll respond as soon as possible.
                            </p>
                        </div>
                        
                        <div>
                            <h3 class="font-semibold text-lg mb-2">Address</h3>
                            <p class="text-gray-700">
                                <?php echo e($business['contact']['address']['full']); ?>
                            </p>
                        </div>
                        
                        <div>
                            <h3 class="font-semibold text-lg mb-2">Opening Hours</h3>
                            <ul class="space-y-1 text-gray-700">
                                <?php
                                $hours = $business['opening_hours'];
                                $day_names = ['Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday', 'Sunday'];
                                $day_keys = ['monday', 'tuesday', 'wednesday', 'thursday', 'friday', 'saturday', 'sunday'];
                                foreach ($day_keys as $index => $day) {
                                    $day_name = $day_names[$index];
                                    if ($hours[$day]['closed']) {
                                        echo "<li>$day_name: Closed</li>";
                                    } else {
                                        echo "<li>$day_name: {$hours[$day]['open']} - {$hours[$day]['close']}</li>";
                                    }
                                }
                                ?>
                            </ul>
                        </div>
                    </div>
                    
                    <!-- Map Embed -->
                    <div class="bg-gray-200 rounded-lg overflow-hidden" style="height: 300px;">
                        <iframe 
                            width="100%" 
                            height="100%" 
                            style="border:0" 
                            loading="lazy" 
                            allowfullscreen
                            src="https://www.google.com/maps/embed/v1/place?key=YOUR_API_KEY&q=<?php echo urlencode($business['contact']['address']['full']); ?>">
                        </iframe>
                        <p class="text-xs text-gray-500 p-2 bg-white">
                            Note: Replace YOUR_API_KEY with your Google Maps API key for the map to display.
                        </p>
                    </div>
                </div>
            </div>
        </div>
    </div>
</section>

<!-- CTA Section -->
<?php include __DIR__ . '/../partials/cta-strip.php'; ?>

<?php include __DIR__ . '/../partials/footer.php'; ?>
